/*! \file
 * \brief S. 65.
 *
 * \author Marussy Kristóf 12. évfolyam     \n
 *         Budapest, Szent István Gimnázium \n
 *         E-mail: kris7topher@gmail.com
 * \note Szükséges fordítóprogram: GCC (g++).
 */

#include <stdint.h>
#include <vector>
#include <string>
#include <sstream>

/*! \def BE
 * \brief A bemeneti fájl neve.
 *
 * Felülírása a fordító kapcsolói között <tt>-DBE='"másik.be"'</tt> formában lehetséges.
 */
#ifndef BE
#   define BE "S65.be"
#endif

/*! \def KI
 * \brief A kimeneti fájl neve.
 *
 * Felülírása a fordító kapcsolói között <tt>-DKI='"másik.ki"'</tt> formában lehetséges.
 */
#ifndef KI
#   define KI "S65.ki"
#endif

/*! \def STDIO
 * \brief Használjon-e a program standard be- és kimenetet.
 *
 * Ha ez a kapcsoló <tt>-DSTDIO=1</tt> formában engedélyezve van,
 * a program a KI és BE állományok helyett a standard be- és kimenetet használja.
 */
#ifndef STDIO
#   define STDIO 0
#endif
#if STDIO > 0
#   include <iostream>
#else
#   include <fstream>
#endif

/*!
 * \brief Az ::N segédtömbben tárolt értékek típúsa.
 *
 * A 64 bites előjel nélküli egészeket a túlcsordulások elkerülése miatt alkalmazzuk.
 */
typedef uint64_t ertek;

/*!
 * \brief Az ::N segédtömb mérete.
 *
 * Az 50-es érték még nem csordul túl, és 12-nél nagyobb blokkméretet is lehetővé tesz.
 */
const size_t kMax = 50;

/*!
 * A felhasznált segédtömb az \f$L_{i,j}\f$ halmazok elemszámaival.
 *
 * \see ::segedtombKiszamol()
 */
std::vector<std::vector<ertek> > N;

/*!
 * \brief Számmá alakít egy betűt.
 *
 * \param[in]  ch       A karakter, az angol ábécéi kisbetűje vagy szóköz (' ').
 * \return A szám.
 */
inline ertek szamma(char ch) {
    if (ch == ' ') {
        return 26;
    } else {
        return ch - 'a';
    }
}

/*!
 * \brief Betűvé alakít egy számot.
 *
 * \param[in]  k        A szám, \f$0 \le k \le 26\f$.
 * \return A karakter.
 */
inline char betuve(ertek k) {
    if (k == 26) {
        return ' ';
    } else {
        return 'a' + static_cast<char>(k);
    }
}

/*!
 * \brief A <em>Segédtömb-Kiszámol</em> eljárás megvalósítása.
 *
 * Felépíti az ::N segédtömböt.
 */
void segedtombKiszamol() {
    N.resize(kMax + 1);
    for (std::vector<std::vector<ertek> >::iterator it = N.begin(); it != N.end(); ++it) {
        it->resize(kMax + 1);
    }
    N[0][0] = 0;
    for (size_t i = 1; i <= kMax; ++i) {
        N[i][i] = 1;
    }
    for (size_t j = 1; j <= kMax; ++j) {
        N[0][j] = N[1][j];
        for (size_t k = j + 1; k <= kMax; ++k) {
            N[k - j][k] = N[k - j + 1][k] + N[k - j - 1][k - 1];
        }
    }
}

/*!
 * \brief A <em>Kódolás-Szóhossz</em> eljárás megvalósítása.
 *
 * \param[in]  nu       A keresett szóhoz tartozó \f$\nu\f$ értéke.
 * \param[out] mu       A keresett szóhoz tartozó \f$\mu\f$ értéke.
 * \return A keresett szóhoz tartozó \f$j\f$ értéke.
 */
inline size_t kodolasSzohossz(ertek nu, ertek & mu) {
    size_t j = 0;
    ertek sum = 0;
    while (sum + N[0][j] < nu) {
        sum += N[0][j];
        ++j;
    }
    mu = nu - sum;
    return j;
}

/*!
 * \brief A <em>Kódolás-Prefixhossz</em> eljárás megvalósítása.
 *
 * \param[in]  j        A keresett szóhoz tartozó \f$j\f$ értéke.
 * \param[in]  mu       A keresett szóhoz tartozó \f$\mu\f$ értéke.
 * \param[out] muKov    A keresett szóhoz tartozó \f$\mu'\f$ értéke.
 * \return A keresett szóhoz tartozó \f$i\f$ értéke.
 */
inline size_t kodolasPrefixhossz(size_t j, ertek mu, ertek & muKov) {
    if (j == 1) {
        return 1;
    }
    size_t i = j;
    while (N[i][j] < mu) {
        --i;
    }
    if (i == j) {
        muKov = mu;
    } else {
        muKov = mu - N[i + 1][j];
    }
    return i;
}

/*!
 * \brief A <em>Kódolás</em> eljárás megvalósítása.
 *
 * \param[in]  nu       A keresett szóhoz tartozó \f$\nu\f$ értéke.
 * \return A keresett szó.
 */
inline std::string kodolas(ertek nu) {
    ertek mu;
    size_t j = kodolasSzohossz(nu, mu);
    size_t i, iElozo = 1;
    std::string str(2 * j, '?');
    std::string::iterator p = str.begin();
    while (j > 0) {
        i = kodolasPrefixhossz(j, mu, mu);
        for (size_t l = 0; l < i + 1 - iElozo; ++l) {
            *(p++) = '<';
        }
        *(p++) = '>';
        iElozo = i;
        --j;
    };
    *p = 0;
    return str;
}

/*!
 * \brief A <em>Dekódolás</em> eljárás megvalósítása.
 *
 * \param[in]  x        A dekódolandó jelsorozat.
 * \return A dekódolt szóhoz tartozó \f$\nu\f$ értéke.
 */
inline ertek dekodolas(const std::string & x) {
    ertek nu = 1;
    size_t j = x.length() / 2;
    size_t i = 0;
    for (std::string::const_iterator p = x.begin(); p != x.end(); ++p) {
        switch (*p) {
            case '<':
                ++i;
                break;
            case '>':
                if (i < j) {
                    nu += N[i + 1][j];
                }
                nu += N[0][--j];
                --i;
                break;
        }
    }
    return nu;
}

/*!
 * \brief Számok sorozatává alakítja a szöveges bemenetet.
 *
 * \param[in]  b        A blokkméret.
 * \param[in]  str      A kódolandó szöveg.
 * \return A kódolandó számok vektora.
 */
std::vector<ertek> felbont(size_t b, const std::string & str) {
    std::vector<ertek> vec(str.length() / b         // A befejezett blokkok száma.
            + ((str.length() % b == 0) ? 0 : 1),    // Az esetleges megkezdett blokk.
            0);
    std::vector<ertek>::iterator it = vec.begin();
    size_t c = 0;
    for (std::string::const_iterator p = str.begin(); p != str.end(); ++p) {
        *it *= 27;
        *it += szamma(*p);
        if (++c == b) {
            // Elértük az átszámított blokk végét, új kezdődik.
            c = 0;
            ++it;
        }
    }
    if (it != vec.end()) {
        // Beleszámítjuk a kitöltő szóközöket az utolsó blokk értékébe.
        for (; c < b; ++c) {
            *it *= 27;
            *it += szamma(' ');
        }
    }
    return vec;
}

/*!
 * \brief Szöveggé alakít vissza egy dekódolt számot.
 *
 * \param[in]  b        A blokkméret.
 * \param[in]  k        A dekódolt szám.
 * \return A dekódolt szövegrészlet.
 */
inline std::string visszaalakit(size_t b, ertek k) {
    std::string szo(b, 'a');
    for(std::string::reverse_iterator p = szo.rbegin(); p != szo.rend(); ++p) {
        *p = betuve(k % 27);
        k /= 27;
    }
    return szo;
}

/*!
 * \brief Elvégzi a kódolási lépéseket.
 *
 * \param[in]  b        A blokkméret.
 * \param[in]  str      A kódolandó szöveg.
 * \param[in]  stream   A stream, ahová a kódolt jelsorozat fog kerülni.
 */
void kodol(size_t b, const std::string & str, std::ostream & stream) {
    std::vector<ertek> vec = felbont(b, str);
    if (vec.size() == 0) {
        // Üres bemenetre üres kimenet.
        return;
    }
    std::vector<ertek>::const_iterator it = vec.begin();
    // Az R listát 0-tól indexeljük, de \f$\nu = 1\f$-hez tartozik az "<>".
    stream << kodolas(*(it++) + 1);
    for (; it != vec.end(); ++it) {
        // Itt is 1-től kezdődik \f$\nu\f$.
        stream << " " << kodolas(*it + 1);
    }
}

/*!
 * \brief Elvégzi a dekódolási lépéseket.
 *
 * \param[in]  b        A blokkméret.
 * \param[in]  str      A dekódolandó jelsorozat.
 * \param[in]  stream   A stream, ahová a dekódolt szöveg fog kerülni.
 */
void dekodol(size_t b, const std::string & str, std::ostream & stream) {
    if (str.length() == 0) {
        // Üres bemenetre üres kimenet.
        return;
    }
    std::stringstream be(str, std::ios_base::in);
    std::string blokk;
    while (!be.eof()) {
        be >> blokk;
        // \f$\nu = 1\f$-hez tartozik az "<>", de az R listát 0-tól indexeljük.
        stream << visszaalakit(b, dekodolas(blokk) - 1);
    }
}

int main(int argc, char ** argv) {
#if STDIO > 0
    std::istream & be(std::cin);
    std::ostream & ki(std::cout);
#else
    std::ifstream be(BE, std::ios_base::in);
    std::ofstream ki(KI, std::ios_base::out | std::ios_base::trunc);
#endif

    segedtombKiszamol();
    // Beolvassuk a blokkméretet.
    size_t b;
    be >> b;
    be.ignore(256, '\n');
    std::string str;
    std::getline(be, str);
    if (str[0] == '<') {    // A bemenet egy kódolt jelsorozat, dekódolunk.
        dekodol(b, str, ki);
    } else {                // A bemenet egy nyílt szöveg, kódolunk.
        kodol(b, str, ki);
    }
    return 0;
}
