﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace TestGenerator
{
	public partial class FormMain : Form
	{
		public class RedInterval : IComparable<RedInterval>
		{
			public int Start, End;

			public RedInterval(int start, int end)
			{
				if (start < end)
				{
					this.Start = start;
					this.End = end;
				}
				else
				{
					this.Start = end;
					this.End = start;
				}
			}

			public int CompareTo(RedInterval other)
			{
				if (Start < other.Start) return -1;
				else if (Start > other.Start) return 1;
				else return 0;
			}
		}

		public class Crossing
		{
			private int position;
			private List<RedInterval> redIntervals = new List<RedInterval>();

			public List<RedInterval> RedIntervals
			{
				get { return redIntervals; }
			}

			public int Position
			{
				get { return position; }
				set { position = value; }
			}
		}

		public class Field
		{
			private int l, m, maxt;
			private List<Crossing> crossings = new List<Crossing>();
			private bool[, ,] possible = null;
			private Point winningEndPoint;
			private int winningSpeed = -1;			
			private double winningTime = double.MaxValue;

			public List<Crossing> Crossings
			{
				get { return crossings; }
			}

			public int MaxT
			{
				get { return maxt; }
				set { maxt = value; }
			}

			public int M
			{
				get { return m; }
				set { m = value; }
			}

			public int L
			{
				get { return l; }
				set { l = value; }
			}

			public int N
			{
				get { return crossings.Count; }
			}

			public bool[, ,] Possible
			{
				get { return possible; }
			}

			public Point WinningEndPoint
			{
				get { return winningEndPoint; }
			}

			public int WinningSpeed
			{
				get { return winningSpeed; }
			}

			public double WinningTime
			{
				get { return winningTime; }
			}

			public void Serialize(TextWriter tw)
			{				
				tw.WriteLine("{0} {1} {2}", L, N, M);
				foreach (Crossing c in crossings)
				{
					StringBuilder sb = new StringBuilder();
					sb.AppendFormat("{0} {1}", c.Position, c.RedIntervals.Count * 2);
					c.RedIntervals.Sort();
					foreach (RedInterval ri in c.RedIntervals)
					{
						sb.AppendFormat(" {0} {1}", ri.Start, ri.End);
					}
					tw.WriteLine(sb.ToString());
				}
			}
			
			public void Deserialize(TextReader tr)
			{
				crossings.Clear();

				String header = tr.ReadLine();
				String[] parameters = header.Split(new char[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
				L = int.Parse(parameters[0]);
				int n = int.Parse(parameters[1]);
				M = int.Parse(parameters[2]);

				for( int i=0; i<n; ++i )
				{
					String line = tr.ReadLine();
					String[] vals = line.Split(new char[]{' '},StringSplitOptions.RemoveEmptyEntries);

					Crossing c = new Crossing();
					c.Position = int.Parse(vals[0]);
					crossings.Add(c);

					int nn = int.Parse(vals[1]);
					int idx = 2;
					for (int j = 0; j < nn/2; ++j)
					{
						int start = int.Parse(vals[idx++]);
						int end = int.Parse(vals[idx++]);
						c.RedIntervals.Add(new RedInterval(start, end));						
					}
				}
			}

			public void Optimize()
			{
				winningTime = Double.MaxValue;

				if (MaxT == 0 || L == 0 || M == 0)
					return;

				possible = new bool[MaxT + 1, L + M, M + 1];
				possible[0, 0, 0] = true;

				for (int t = 1; t <= MaxT; ++t)
				{
					for (int y = 0; y <= L + M - 1; ++y)
					{
						for (int v = 0; v <= M; ++v)
						{
							for (int dv = -1; dv <= 1; ++dv)
							{
								int pv = v - dv;
								int py = y - v;

								if (py >= L || py < 0 || pv < 0 || pv > M)
									continue;

								if (!possible[t - 1, py, pv])
									continue;
								
								bool blocked = false;
								foreach (Crossing c in Crossings)
								{
									if (c.Position < py || c.Position > y)
										continue;

									foreach (RedInterval ri in c.RedIntervals)
									{
										if (v == 0)
										{
											if (y == c.Position && t > ri.Start && t <= ri.End)
											{
												blocked = true;
												break;
											}
										}
										else
										{
											if (y >= c.Position && py < c.Position && (t - 1) >= ri.Start && t <= ri.End)
											{
												blocked = true;
												break;
											}
										}
									}

									if (blocked) break;
								}

								if (!blocked)
									possible[t, y, v] = true;
							}

							if (possible[t, y, v] && y >= L && v != 0 && v > (y - L))
							{
								double time = t - (double)(y - L) / v;
								if (time < winningTime)
								{
									winningTime = time;
									winningSpeed = v;
									winningEndPoint = new Point(t, y);
								}
							}
						}
					}
				}				
			}
		}

		public enum MouseAction
		{
			None,
			NewInterval,
			MoveInterval
		}

		Field field = new Field();
		Size graphCellSize = new Size(45, 15);
		int graphMargin = 30;
		Point cursorPos;
		Point? newIntervalStartPos = null;
		Point lastCursorPos;
		RedInterval selectedInterval = null;
		MouseAction action = MouseAction.None;		

		public FormMain()
		{
			InitializeComponent();
		}

		
		private Point plotPoint(Point p)
		{
			Rectangle viewport = new Rectangle(new Point(0, 0), picGraph.Size);
			viewport.Inflate(-graphMargin, -graphMargin);
			return new Point(viewport.Left + graphCellSize.Width * p.X, viewport.Bottom - graphCellSize.Height * p.Y);
		}

		private Point unplotPoint(Point p)
		{
			Rectangle viewport = new Rectangle(new Point(0, 0), picGraph.Size);
			viewport.Inflate(-graphMargin, -graphMargin);
			return new Point((p.X - viewport.Left + graphCellSize.Width/2) / graphCellSize.Width, (viewport.Bottom - p.Y + graphCellSize.Height / 2) / graphCellSize.Height);
		}

		private PointF plotPointF(Point p)
		{
			Rectangle viewport = new Rectangle(new Point(0, 0), picGraph.Size);
			viewport.Inflate(-graphMargin, -graphMargin);
			return new PointF(viewport.Left + graphCellSize.Width * p.X, viewport.Bottom - graphCellSize.Height * p.Y);
		}

		private void update(bool recalculate)
		{
			if (recalculate)
			{
				field.Optimize();

				if (field.WinningTime != double.MaxValue)
				{
					int frac = 0;
					int i = field.WinningEndPoint.X;
					if (field.WinningEndPoint.Y != field.L)
					{
						--i;
						frac = field.L - (field.WinningEndPoint.Y - field.WinningSpeed);
					}
					lblBestTime.Text = "Idő: " + i + " " + frac + "/" + field.WinningSpeed + " s";
				}
				else
				{
					lblBestTime.Text = "Idő: N/A";
				}
			}

			picGraph.Invalidate();
		}

		private void picGraph_Paint(object sender, PaintEventArgs e)
		{
			Graphics g = e.Graphics;
			g.Clear(Color.White);
			
			using (Pen blackPen = new Pen(Color.Black))
			using (Pen grayPen = new Pen(Color.LightGray))
			using (Pen bluePen = new Pen(Color.Blue))
			using (Pen cyanPen = new Pen(Color.DarkCyan))
			using (Pen darkGrayPen = new Pen(Color.DarkGray))
			using (Brush br = new SolidBrush(Color.Black))
			using (Brush blueBrush = new SolidBrush(Color.DarkBlue))
			{
				// Draw axes
				blackPen.EndCap = System.Drawing.Drawing2D.LineCap.ArrowAnchor;
				g.DrawLine(blackPen, plotPoint(new Point(0, 0)), plotPoint(new Point(0, field.L + 1)));				
				g.DrawLine(blackPen, plotPoint(new Point(0, 0)), plotPoint(new Point(field.MaxT + 1,0)));
				blackPen.EndCap = System.Drawing.Drawing2D.LineCap.NoAnchor;

				// Draw grid and markings
				for( int x=1; x<=field.MaxT; ++x )
				{
					Point origin = plotPoint(new Point(x,0));
					Point a = origin, b = origin, c = origin;
					a.Offset(0,-5);
					b.Offset(0,5);
					c.Offset(-5, 8);					
					
					g.DrawLine(grayPen, plotPoint(new Point(x, 0)), plotPoint(new Point(x, field.L + 1)));
					if (x % 5 == 0)
					{
						g.DrawString(x.ToString(), Font, br, c.X, c.Y);
						g.DrawLine(darkGrayPen, plotPoint(new Point(x, 0)), plotPoint(new Point(x, field.L + 1)));
					}

					g.DrawLine(blackPen, a, b);
				}

				for (int y = 1; y <= field.L; ++y)
				{
					Point origin = plotPoint(new Point(0, y));
					Point a = origin, b = origin, c = origin;
					a.Offset(-5,0);
					b.Offset( 5,0);
					c.Offset( -20,-5);

					g.DrawLine(grayPen, plotPoint(new Point(0,y)), plotPoint(new Point(field.MaxT + 1,y)));
					if (y % 5 == 0)
					{
						g.DrawString(y.ToString(), Font, br, c.X, c.Y);
						g.DrawLine(darkGrayPen, plotPoint(new Point(0, y)), plotPoint(new Point(field.MaxT + 1, y)));
					}

					g.DrawLine(blackPen, a, b);
				}

				// Draw endline
				cyanPen.Width = 2;
				g.DrawLine(cyanPen, plotPoint(new Point(0, field.L)), plotPoint(new Point(field.MaxT, field.L)));
		
				// Draw cursor
				Point m = plotPoint(cursorPos);
				m.Offset(-5,-5);
				g.DrawEllipse(darkGrayPen, new Rectangle(m, new Size(10, 10)));

				// Draw intervals
				blackPen.Width = 3;
				foreach (Crossing c in field.Crossings)
				{
					foreach (RedInterval ri in c.RedIntervals)
					{
						Point start = new Point(ri.Start, c.Position);
						Point end = new Point(ri.End, c.Position);

						if (ri == selectedInterval)
						{
							blackPen.Width = 5;
						}
						else
						{
							blackPen.Width = 3;
						}

						Point startOffset = plotPoint(start);
						startOffset.Offset(5, 0);
						g.DrawLine(blackPen, startOffset, plotPoint(end));

						blackPen.Width = 1;

						Point startDot = plotPoint(start);
						startDot.Offset(-5, -5);
						g.DrawEllipse(blackPen, new Rectangle(startDot, new Size(10, 10)));

						Point endDot = plotPoint(end);
						endDot.Offset(-5, -5);
						g.FillEllipse(br, new Rectangle(endDot, new Size(10, 10)));
					}
				}

				// Draw active intervals
				blackPen.Width = 5;
				if (action == MouseAction.NewInterval)
				{
					Point start = newIntervalStartPos.Value;
					Point end = cursorPos;

					g.DrawLine(blackPen, plotPoint(start), plotPoint(end));
				}

				if (field.Possible == null) return;

				// Draw best path
				if (field.WinningTime != double.MaxValue)
				{
					cyanPen.Width = 3;
					drawPath(field.WinningEndPoint, field.WinningSpeed, g, cyanPen);
				}

				// Draw path
				bluePen.Width = 3;
				drawPath(cursorPos, -1, g, bluePen);				

				// Draw possible states
				for (int t = 0; t <= field.MaxT; ++t)
				{
					for (int y = 0; y <= field.L + field.M - 1; ++y)
					{
						bool poss = false;
						for (int v = 0; v <= field.M; ++v)
						{
							poss = poss || field.Possible[t, y, v];
						}

						if (poss)
						{
							Point pp = plotPoint(new Point(t, y));
							pp.Offset(-2, -2);
							g.FillEllipse(blueBrush, new Rectangle(pp, new Size(4, 4)));
						}
					}
				}
			}
		}

		private void drawPath(Point endPos, int endSpeed, Graphics g, Pen pen)
		{
			if (endPos.X < 0 || endPos.Y < 0 || endPos.X > field.MaxT || endPos.Y > field.L + field.M - 1)
				return;

			Point lastCoord = endPos;
			int lastSpeed = -1;
			if (endSpeed == -1)
			{
				for (int v = 0; v <= field.M; ++v)
				{
					if (field.Possible[lastCoord.X, lastCoord.Y, v]) lastSpeed = v;
				}
				if (lastSpeed == -1)
					return;
			}
			else
			{
				lastSpeed = endSpeed;
			}
			
			while (lastCoord.X > 0)
			{
				Point coord = new Point(lastCoord.X - 1, lastCoord.Y - lastSpeed);
				int speed = -1;
				for (int dv = -1; dv <= 1; ++dv)
				{
					int candSpeed = lastSpeed + dv;
					if (candSpeed < 0 || candSpeed > field.M || coord.Y < 0) continue;
					if (field.Possible[coord.X, coord.Y, candSpeed]) speed = candSpeed;
				}

				if (speed == -1) break;

				g.DrawLine(pen, plotPoint(lastCoord), plotPoint(coord));
				lastSpeed = speed;
				lastCoord = coord;
			}
			
		}

		private void txtL_TextChanged(object sender, EventArgs e)
		{
			try
			{
				field.L = int.Parse(txtL.Text);
				update(true);
			}
			catch (Exception ee) {}
		}		

		private void txtM_TextChanged(object sender, EventArgs e)
		{
			try
			{
				field.M = int.Parse(txtM.Text);
				update(true);
			}
			catch (Exception ee) { }
		}

		private void txtMaxT_TextChanged(object sender, EventArgs e)
		{
			try
			{
				field.MaxT = int.Parse(txtMaxT.Text);
				update(true);
			}
			catch (Exception ee) { }
		}

		private void button1_Click(object sender, EventArgs e)
		{
			SaveFileDialog dlg = new SaveFileDialog();
			if (dlg.ShowDialog() == DialogResult.OK)
			{
				using (StreamWriter sw = new StreamWriter(dlg.FileName, false, Encoding.GetEncoding("windows-1250")))
				{
					field.Serialize(sw);
				}
			}
		}

		private void Form_Load(object sender, EventArgs e)
		{
			txtL.Text = "20";
			txtM.Text = "30";
			txtMaxT.Text = "15";
		}

		private void picGraph_MouseMove(object sender, MouseEventArgs e)
		{
			lastCursorPos = cursorPos;
			cursorPos = unplotPoint(e.Location);

			if (cursorPos == lastCursorPos)
				return;

			if (action == MouseAction.MoveInterval)
			{
				selectedInterval.Start += cursorPos.X - lastCursorPos.X;
				selectedInterval.End += cursorPos.X - lastCursorPos.X;
				update(true);
			}
			else
			{
				update(false);
			}
			
		}

		private void picGraph_MouseDown(object sender, MouseEventArgs e)
		{
			Point downPoint = unplotPoint(e.Location);
			foreach (Crossing c in field.Crossings)
			{
				foreach (RedInterval ri in c.RedIntervals)
				{
					Point start = new Point(ri.Start, c.Position);
					Point end = new Point(ri.End, c.Position);

					if (downPoint.Y == start.Y && downPoint.X >= start.X && downPoint.X <= end.X)
					{
						if ((e.Button & MouseButtons.Left) != 0)
						{
							selectedInterval = ri;
							action = MouseAction.MoveInterval;
							update(false);
						}
						else
						{
							c.RedIntervals.Remove(ri);
							if (c.RedIntervals.Count == 0)
								field.Crossings.Remove(c);
							txtN.Text = field.Crossings.Count.ToString();
							update(true);
						}
						return;
					}
				}
			}

			action = MouseAction.NewInterval;
			newIntervalStartPos = cursorPos;
			update(false);
		}

		private void picGraph_MouseUp(object sender, MouseEventArgs e)
		{
			if (action == MouseAction.NewInterval)
			{
				Point endPoint = unplotPoint(e.Location);

				if (endPoint.X != newIntervalStartPos.Value.X)
				{
					Crossing c = null;
					foreach (Crossing cc in field.Crossings)
					{
						if (cc.Position == newIntervalStartPos.Value.Y)
						{
							c = cc;
							break;
						}
					}
					if (c == null)
					{
						c = new Crossing();
						field.Crossings.Add(c);
						c.Position = newIntervalStartPos.Value.Y;
					}

					c.RedIntervals.Add(new RedInterval(newIntervalStartPos.Value.X, endPoint.X));
					txtN.Text = field.Crossings.Count.ToString();
					newIntervalStartPos = null;
					action = MouseAction.None;
					update(true);
				}
				else
				{
					newIntervalStartPos = null;
					action = MouseAction.None;
					update(false);
				}
			}
			else if (action == MouseAction.MoveInterval)
			{
				selectedInterval = null;
				action = MouseAction.None;
				update(true);
			}
		}

		private void button3_Click(object sender, EventArgs e)
		{
			field.Crossings.Clear();
			update(true);
		}

		private void button2_Click(object sender, EventArgs e)
		{
			OpenFileDialog dlg = new OpenFileDialog();
			if (dlg.ShowDialog() == DialogResult.OK)
			{
				using (StreamReader sr = new StreamReader(dlg.FileName, Encoding.GetEncoding("windows-1250")))
				{
					field.Deserialize(sr);
				}
			}

			txtL.Text = field.L.ToString();
			txtM.Text = field.M.ToString();
			txtN.Text = field.Crossings.Count.ToString();

			update(true);
		}

		private void Form_SizeChanged(object sender, EventArgs e)
		{
			update(false);
		}

		private void button4_Click(object sender, EventArgs e)
		{
			Field big = new Field();
			big.M = 30;
			big.L = 5000;

			Random r = new Random();
			
			for( int i=0; i<100; ++i )
			{
				Crossing c = new Crossing();
				c.Position = (i+1)*30;

				int x = 1;
				for( int j=0; j<5; ++j)
				{
					RedInterval ri = new RedInterval(0,0);
					ri.Start = x;
					x += r.Next(1000);
					ri.End = x;
					x += 10 + r.Next(1000);

					c.RedIntervals.Add(ri);
				}

				big.Crossings.Add(c);
			}

			SaveFileDialog dlg = new SaveFileDialog();
			if (dlg.ShowDialog() == DialogResult.OK)
			{
				using (StreamWriter sw = new StreamWriter(dlg.FileName, false, Encoding.GetEncoding("windows-1250")))
				{
					big.Serialize(sw);
				}
			}
		}
	}
}
